<?php

use Carbon\Carbon;
use App\Models\Link;
use App\Models\Work;
use App\Models\Piece;
use App\Models\Visit;
use App\Models\Insight;
use App\Models\Service;
use App\Models\Experiment;
use Illuminate\Http\Request;
use App\Services\AnalyticsService;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Route;
use App\Http\Controllers\Backend\LinkController as BackendLinkController;
use App\Http\Controllers\Backend\WorkController as BackendWorkController;
use App\Http\Controllers\Backend\PieceController as BackendPieceController;
use App\Http\Controllers\Backend\InsightController as BackendInsightController;
use App\Http\Controllers\Backend\ServiceController as BackendServiceController;
use App\Http\Controllers\Backend\ExperimentController as BackendExperimentController;


Route::name('frontend.')->group(function () {
    
    // --- HOME PAGE ---
    Route::get('/', function (Request $request, AnalyticsService $analyticsService) {
        // Panggil Service di sini (DI via closure argument)
        $analyticsService->recordVisit($request); 
        
        $services = Service::all();
        $links = Link::all();
        return view('frontend.welcome', compact('services', 'links'));
    })->name('home');

    // --- WORKS ---
    Route::get('/works', function (Request $request, AnalyticsService $analyticsService) {
        $analyticsService->recordVisit($request);
        
        $works = Work::all();
        return view('frontend.works', compact('works'));
    })->name('works');
    
    // Asumsi works.show adalah halaman detail, tambahkan pelacakan di sini juga
    Route::get('/works/show', function (Request $request, AnalyticsService $analyticsService) {
        $analyticsService->recordVisit($request);

        $works = Work::all(); // Biasanya di sini hanya mengambil 1 item, tapi mengikuti kode Anda
        return view('frontend.works', compact('works'));
    })->name('works.show');

    // --- LABS ---
    Route::get('/labs', function (Request $request, AnalyticsService $analyticsService) {
        $analyticsService->recordVisit($request);
        
        $experiments = Experiment::all();
        return view('frontend.labs', compact('experiments'));
    })->name('labs');
    
    // --- INSIGHTS ---
    Route::get('/insights', function (Request $request, AnalyticsService $analyticsService) {
        $analyticsService->recordVisit($request);
        
        $insights = Insight::all();
        return view('frontend.insights', compact('insights'));
    })->name('insights');

    // --- SERVICES ---
    Route::get('/services', function (Request $request, AnalyticsService $analyticsService) {
        $analyticsService->recordVisit($request);
        
        $services = Service::all();
        return view('frontend.services', compact('services'));
    })->name('services');

    // --- PIECES CATEGORIES ---
    Route::get('/pieces', function (Request $request, AnalyticsService $analyticsService) {
        $analyticsService->recordVisit($request);
        
        $categories = Piece::distinct()->pluck('category');
        return view('frontend.pieces.list', compact('categories'));
    })->name('pieces.categories');

    // --- PIECES DETAIL/VIEW ---
    Route::get('/pieces/{category}', function (Request $request, string $category, AnalyticsService $analyticsService) {
        $analyticsService->recordVisit($request);
        
        $pieces = Piece::where('category', $category)->orderBy('title')->get();
        return view('frontend.pieces.view', compact('pieces', 'category'));
    })->name('pieces.show_by_category');
});

Route::middleware('dashboard.guest')->group(function () {
    Route::get('access', function () {
        return view('access');
    })->name('access');

    Route::post('/access', function (Request $request) {
        // Cukup ambil satu kredensial, yaitu 'password' (yang berperan sebagai PIN)
        $credentials = $request->only('password');

        // Lakukan validasi PIN
        // Sebaiknya gunakan variabel .env yang berbeda untuk PIN, misal 'DASHBOARD_PIN'
        if ($credentials['password'] === env('DASHBOARD_PIN')) {
            // Log in pengguna secara manual (menggunakan session)
            session()->put('logged_in', true);
            return redirect()->route('backend.dashboard');
        }

        // Kembali dengan pesan kesalahan jika PIN tidak valid
        return back()->withErrors(['password' => 'Invalid PIN.']);
    })->name('access');
});
Route::middleware('dashboard.auth')->group(function () {
    Route::post('/revoke', function (Request $request) {
        // Hapus otentikasi dan sesi
        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('access');
    })->name('revoke');
});

Route::name('backend.')->middleware('dashboard.auth')->prefix('auth')->group(function () {
   
    Route::get('/', function (Request $request) {
    
        // --- 1. Ambil Statistik Kunci ---
        $totalHits = Visit::count();
        
        // BARU: Hitung jumlah hari unik yang tercatat di database.
        // Ini lebih akurat untuk data yang sedikit dan yang ingin tahu rata-rata per hari yang memiliki data.
        $uniqueDaysCount = Visit::distinct(DB::raw('DATE(created_at)'))->count();
        
        // Tentukan pembagi (divisor)
        $divisor = $uniqueDaysCount ?: 1; // Jika tidak ada data, gunakan 1 untuk menghindari pembagian dengan nol
        
        // Hitung Rata-rata Harian
        // Jika data hanya 4 dan tercatat hanya pada 1 hari unik, maka 4 / 1 = 4.
        $dailyAverage = $totalHits > 0 ? round($totalHits / $divisor, 2) : 0; 
        
        // Data lama (sebelumnya): Ini hanya dibutuhkan jika Anda ingin tahu rata-rata sejak hari pertama kunjungan
        // $firstVisitDate = Visit::min('created_at');
        // $startDate = $firstVisitDate ? Carbon::parse($firstVisitDate) : now();
        // $days = $startDate->diffInDays(now()) ?: 1; // Jika ini 1 hari, dan totalHits 4, hasilnya 4. Jika 30 hari, hasilnya 0.13.
    
        $uniqueVisitors = Visit::distinct('session_id')->count();
        
        // --- 2. Data Kunjungan Harian (Untuk Chart) ---
        $dailyVisits = Visit::select(
            DB::raw('DATE(created_at) as date'),
            DB::raw('count(*) as count')
        )
        ->groupBy('date')
        ->orderBy('date')
        ->limit(30)
        ->get();
        
        // --- 3. Data Geolokasi (Untuk Maps/Heatmap) ---
        $geoData = Visit::select('country', DB::raw('count(*) as count'))
                         ->whereNotNull('country')
                         ->groupBy('country')
                         ->get()
                         ->pluck('count', 'country')
                         ->toArray();
                         
        // --- 4. Riwayat Kunjungan Terbaru ---
        $recentVisits = Visit::orderByDesc('created_at')
                             ->select('city', 'country', 'url', 'created_at')
                             ->limit(5)
                             ->get();
    
        return view('backend.dashboard', compact(
            'totalHits', 
            'uniqueVisitors', 
            'dailyVisits', 
            'geoData', 
            'recentVisits',
            'dailyAverage'
        ));
    })->name('dashboard');
    
    Route::resource('works', BackendWorkController::class);
    Route::resource('insights', BackendInsightController::class);
    Route::resource('services', BackendServiceController::class);
    Route::resource('experiments', BackendExperimentController::class);
    Route::resource('links', BackendLinkController::class);
    Route::resource('pieces', BackendPieceController::class);
    
});