<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\Work;
use App\Models\Media;
use App\Traits\HasMedia;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class WorkController extends Controller
{
    use HasMedia;

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $works = Work::all();
        return view('backend.works.index', compact('works'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        return view('backend.works.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'subtitle' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'tags' => 'nullable|string',
            'url' => 'nullable|url'
        ]);

        try {
            DB::beginTransaction();

            $tagsArray = $request->input('tags') ? array_map('trim', explode(',', $request->input('tags'))) : [];

            $work = Work::create([
                'title' => $request->input('title'),
                'subtitle' => $request->input('subtitle'),
                'description' => $request->input('description'),
                'tags' => $tagsArray,
                'slug' => Str::slug($request->input('title')),
                'url' => $request->input('url'),
            ]);

            $this->handleMedia($work, $request, 'media_files');

            DB::commit();

            return redirect()->route('backend.works.index')->with('success', 'Work created successfully!');
        } catch (Exception $e) {
            DB::rollBack();
            Log::error('Error creating work: ' . $e->getMessage());
            return redirect()->back()->withInput()->with('error', 'Failed to create work. Please try again.');
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Work  $work
     * @return \Illuminate\View\View
     */
    public function show(Work $work)
    {
        $work->load('media');
        return view('backend.works.show', compact('work'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Work  $work
     * @return \Illuminate\View\View
     */
    public function edit(Work $work)
    {
        $work->load('media');
        return view('backend.works.edit', compact('work'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Work  $work
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, Work $work)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'subtitle' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'tags' => 'nullable|string',
            'url' => 'nullable|url',
            'media_files.*' => 'nullable|file|mimes:jpeg,png,jpg,gif,mp4,mov,ogg,webm|max:20480',
        ]);

        try {
            DB::beginTransaction();

            $tagsArray = $request->input('tags') ? array_map('trim', explode(',', $request->input('tags'))) : [];

            $work->title = $request->input('title');
            $work->subtitle = $request->input('subtitle');
            $work->description = $request->input('description');
            $work->tags = $tagsArray;
            $work->url = $request->input('url');

            if ($work->isDirty('title')) {
                $work->slug = Str::slug($request->input('title'));
            }
            $work->save();

            $this->handleMedia($work, $request, 'media_files');

            DB::commit();

            return redirect()->route('backend.works.index')->with('success', 'Work updated successfully!');
        } catch (Exception $e) {
            DB::rollBack();
            Log::error('Error updating work: ' . $e->getMessage());
            return redirect()->back()->withInput()->with('error', 'Failed to update work. Please try again.');
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Work  $work
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Work $work)
    {
        try {
            DB::beginTransaction();

            $work->media()->each(function ($media) {
                Storage::disk('public')->delete($media->file_path);
                $media->delete();
            });

            $work->delete();

            DB::commit();
            return redirect()->route('backend.works.index')->with('success', 'Work deleted successfully!');
        } catch (Exception $e) {
            DB::rollBack();
            Log::error('Error deleting work: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to delete work. Please try again.');
        }
    }
}
