<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\Link;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class LinkController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $links = Link::orderBy('order')->get();
        return view('backend.links.index', compact('links'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        return view('backend.links.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:50|unique:links,name',
            'url' => 'required|url|unique:links,url',
            'icon' => 'nullable|string|max:50',
            'order' => 'nullable|integer',
        ]);

        try {
            Link::create($request->all());

            return redirect()->route('backend.links.index')->with('success', 'Link created successfully! ✨');
        } catch (Exception $e) {
            Log::error('Error creating link: ' . $e->getMessage());
            return redirect()->back()->withInput()->with('error', 'Failed to create link. Please try again.');
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Link  $link
     * @return \Illuminate\View\View
     */
    public function edit(Link $link)
    {
        return view('backend.links.edit', compact('link'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Link  $link
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, Link $link)
    {
        $request->validate([
            'name' => 'required|string|max:50|unique:links,name,' . $link->id,
            'url' => 'required|url|unique:links,url,' . $link->id,
            'icon' => 'nullable|string|max:50',
            'order' => 'nullable|integer',
        ]);

        try {
            $link->update($request->all());

            return redirect()->route('backend.links.index')->with('success', 'Link updated successfully! 🚀');
        } catch (Exception $e) {
            Log::error('Error updating link: ' . $e->getMessage());
            return redirect()->back()->withInput()->with('error', 'Failed to update link. Please try again.');
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Link  $link
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Link $link)
    {
        try {
            $link->delete();
            return redirect()->route('backend.links.index')->with('success', 'Link deleted successfully! 🗑️');
        } catch (Exception $e) {
            Log::error('Error deleting link: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to delete link. Please try again.');
        }
    }
}
